#ifndef __CMusicMath__
#define __CMusicMath__

#include "CTimeSignature.hpp"
#include <Basics/CString.hpp>

//	===========================================================================

using Exponent::Music::CTimeSignature;
using Exponent::Basics::CString;

//	===========================================================================

namespace Exponent
{
	namespace Music
	{
		/**
		 * @class CMusicMath CMusicMath.hpp
		 * @brief Lots of calculations with musical relevance : ** ITS SCAL AS A CLASS! **
		 *
		 * @date 07/05/2005
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CMusicMath.hpp,v 1.4 2007/02/08 21:08:09 paul Exp $
		 */
		class CMusicMath
		{
		public:

//	===========================================================================

			const static double CMUSIC_MATH_FREQUENCY[];					/**< Frequencies for tempo divisions */
			const static double CMUSIC_MATH_TIME[];							/**< Timing information for tempo divisions */
			const static double CMUSIC_MATH_SONG_PHASE[];					/**< Song phase for computing offset */
			const static double CMUSIC_MATH_SONG_POSITION[];				/**< Position modifier for song phase */

//	===========================================================================

			/**
			 * @enum ETempoDivision
			 * @brief Enumeration of tempo divisions
			 */
			enum ETempoDivision
			{
				// Straight
				e_sixtyFourBars = 0,		/**< 64 bar sync */
				e_thirtyTwoBars,			/**< 32 bar sync */
				e_sixteenBars,				/**< 16 bar sync */
				e_eightBars,				/**< 8 bar sync */
				e_fourBars,					/**< 4 bar sync */
				e_twoBars,					/**< 2 bar sync */
				e_oneBar, 					/**< 1 bar sync */
				e_halfBar, 					/**< 1/2 bar sync */
				e_quarterBar,				/**< 1/4 bar sync */
				e_eigthBar, 				/**< 1/8 bar sync */
				e_sixteenthBar, 			/**< 1/16 bar sync */
				e_thirtySecondBar, 			/**< 1/32 bar sync */
				e_sixtyFourthBar,			/**< 1/64 bar sync */

				// Triplet
				e_sixtyFourBarsTriplet,		/**< 64T bar sync */
				e_thirtyTwoBarsTriplet,		/**< 32T bar sync */
				e_sixteenBarsTriplet,		/**< 16T bar sync */
				e_eightBarsTriplet,			/**< 8T bar sync */
				e_fourBarsTriplet, 			/**< 4T bar sync */
				e_twoBarsTriplet, 			/**< 2T bar sync */
				e_oneBarTriplet,  			/**< 1T bar sync */
				e_halfBarTriplet, 			/**< 1/2T bar sync */
				e_quarterBarTriplet, 		/**< 1/4T bar sync */
				e_eigthBarTriplet, 			/**< 1/8T bar sync */
				e_sixteenthBarTriplet, 		/**< 1/16T bar sync */
				e_thirtySecondBarTriplet, 	/**< 1/32T bar sync */
				e_sixtyFourthBarTriplet,	/**< 1/64T bar sync */

				// Dotted
				e_sixtyFourBarsDotted,		/**< 64. bar sync */
				e_thirtyTwoBarsDotted,		/**< 32. bar sync */
				e_sixteenBarsDotted,		/**< 16. bar sync */
				e_eightBarsDotted,			/**< 8. bar sync */
				e_fourBarsDotted,			/**< 4. bar sync */
				e_twoBarsDotted,			/**< 2. bar sync */
				e_oneBarDotted,				/**< 1. bar sync */
				e_halfBarDotted,			/**< 1/2. bar sync */
				e_quarterBarDotted,			/**< 1/4. bar sync */
				e_eigthBarDotted,			/**< 1/8. bar sync */
				e_sixteenthBarDotted,		/**< 1/16. bar sync */
				e_thirtySecondBarDotted,	/**< 1/32. bar sync */
				e_sixtyFourthBarDotted,		/**< 1/64. bar sync */

				e_numTempoDivisions			/**< Total number of tempo divisions */
			};

//	===========================================================================

			/**
			 * Get a rate in milliseconds for a given tempo division of the bpm
			 * @param bpm The beats per minute
			 * @param tempoDivision The requested tempo division
			 * @param timeSignature The time signature of the tune
			 * @retval The rate in milliseconds required
			 */
			static double getRateInMilliseconds(const double bpm, const ETempoDivision tempoDivision, const CTimeSignature &timeSignature);

			/**
			 * Get a rate in hertz for a given tempo division of the bpm
			 * @param bpm The beats per minute
			 * @param tempoDivision The requested tempo division
			 * @retval The rate in hertz required
			 */
			static double getRateInHertz(const double bpm, const ETempoDivision tempoDivision);

			/**
			 * Get the closest division for a given frequency and bpm
			 * @param bpm The beats per minute of the song
			 * @param frequency The frequency to convert to a division
			 * @retval ETempoDivision The closest tempo division
			 */
			static ETempoDivision getClosestTempoDivisionToRateInHertz(const double bpm, const double frequency);

			/**
			 * Get a phase position for a given tempo division
			 * @param tempoDivision The tempo division
			 * @retval double The phase reset position
			 */
			static double getPhase(const ETempoDivision tempoDivision);

			/**
			 * Get a song phase postion for a given tempo division
			 * @param tempoDivision The tempo division
			 * @retval double The phase position
			 */
			static double getSongPhasePosition(const ETempoDivision tempoDivision);

//	===========================================================================

			/**
			 * Get number of samples in a specified length (length in seconds
			 * @param sampleRate The number of samples per second
			 * @param lengthInSeconds The length of the buffer in seconds
			 * @retval double The number of samples in the specified length of time
			 */
			static double getNumberOfSamples(const double sampleRate, const double lengthInSeconds);

//	===========================================================================

			/**
			 * Given a frequency what is the closest midi note
			 * @param frequency The frequency to convert
			 * @retval long The closest midi note to the frequency
			 */
			static long getClosestMidiNote(const double frequency);

			/**
			 * Get a frequency for a midi note
			 * @param note The midi note to get the frequency for
			 * @retval double The frequency, or 0.0 if index out of range
			 */
			static double getNoteFrequency(const long note);

			/**
			 * Get a detailed frequency
			 * @param note The midi note
			 * @param octaveDetune The number of octaves the note is detuned by
			 * @param semiDetune The number of semi tones the note is detuned by
			 * @param fineDetune The number of cents the note is detuned by
			 * @retval double The frequency, or 0.0 if error
			 */
			static double getNoteFrequency(const long note, const long octaveDetune, const long semiDetune, const long fineDetune);

//	===========================================================================

			/**
			 * Given an original bpm and a desired bpm how much do you have to shift it by in semitones
			 * @param originalBpm The original Bpm
			 * @param newBpm The destination Bpm
			 * @retval double The shift in semi tones
			 */
			static double getTempoNoteShift(const double originalBpm, const double newBpm);

			/**
			 * To go from one bpm to antoher what is the timestretch required (%)
			 * @param originalBpm The original Bpm
			 * @param newBpm The destination Bpm
			 * @retval double The amount of time stretch required in %
			 */
			static double getTimeStretch(const double originalBpm, const double newBpm);

			/**
			 * Given an original bpm and the amount shifted by, what is the new bpm
			 * @param shift The shift amount in semi tones
			 * @param bpm The original bpm
			 * @retval double The new bpm
			 */
			static double getNewTempoPitchShift(const double shift, const double bpm);

			/**
			 * Given a length (seconds) and time sig information what is the bpm
			 * @param length The length in seconds
			 * @param timeSignature The time signature
			 * @param numberOfBeats The number of beats in the section of music
			 * @retval double The bpm
			 */
			static double getBpmFromTime(const double length, const CTimeSignature &timeSignature, const long numberOfBeats = 4);

			/**
			 * Given a bpm and time sig information what is the length
			 * @param bpm The bpm
			 * @param timeSignature The time signature
			 * @param numberOfBeats The number of beats in the section of music
			 * @retval double The length in seconds
			 */
			static double getTimeFromBpm(const double bpm, const CTimeSignature &timeSignature, const long numberOfBeats = 4);

//	===========================================================================

			/**
			 * Get a division string
			 * @param theString The string that will  be filled with the tempo division
			 * @param tempoDivision The tempo division
			 */
			static void getADivisionString(CString &theString, const ETempoDivision tempoDivision);

//	===========================================================================

		private:

//	===========================================================================

			/**
			 * @enum ETempoDivisionType
			 * @brief Different types of division, re swing
			 */
			enum ETempoDivisionType
			{
				e_straight = 0,				/**< Straight time */
				e_triplet,					/**< Triplet time */
				e_dotted,					/**< Dotted time */
			};

//	===========================================================================

			/**
			 * @class CFrequencyTable CMusicMath.h
			 * @brief A class to calculate the midi/frequency table
			 */
			class CFrequencyTable
			{
			public:

				/**
				 * Construction
				 */
				CFrequencyTable()
				{
					// 12th root of 2
					const double kappa = 1.059463094359;

					// a
					double frequency = 6.875;

					// b
					frequency *= kappa;

					// bb
					frequency *= kappa;

					// c, frequency of midi note 0
					frequency *= kappa;

					// Generate 128 midi notes
					for (long i = 0; i < 128; i++)
					{
						m_frequencyTable[i] = frequency;
						frequency *= kappa;
					}
				}

				/**
				 * Assignment operator
				 * @param table The table to copy
				 * @retval CFrequencyTable& A reference to this
				 */
				CFrequencyTable &operator = (const CFrequencyTable &table)
				{
					// copy 128 midi notes
					for (long i = 0; i < 128; i++)
					{
						m_frequencyTable[i] = table.m_frequencyTable[i];
					}
					return *this;
				}

				/**
				 * Get an element of the frequency table
				 * @param index The index to get
				 * @retval double The frequency or 0.0 if index out of range
				 */
				double getFrequency(const long index) const
				{
					if (index >= 0 && index < 128)
					{
						return m_frequencyTable[index];
					}
					return 0.0;
				}

//	===========================================================================

			protected:

//	===========================================================================

				double m_frequencyTable[128];								/**< The frequency information */
			};

//	===========================================================================

			/**
			 * Get an adjusted tempo division -> in the range of stright time
			 * @param division The division to convert
			 * @param type The type of division to compute
			 * @retval ETempoDivision The adjusted tempo division in the range of stright time
			 */
			static ETempoDivision getAdjustedTempoDivision(const ETempoDivision division, ETempoDivisionType &type);

			/**
			 * Convert a value to dotted or triplet time
			 * @param value The value to convert
			 * @param type The type of division to compute
			 * @retval double The converted value
			 */
			static double convertValueToTempoDivisionType(const double value, const ETempoDivisionType type);

//	===========================================================================

			const static CFrequencyTable CMUSIC_MATH_FREQUENCY_TABLE;		/**< The frequency table */

			const static double CMUSIC_MATH_TRIPLET_TIME;					/**< Triplet time multiplier */
			const static double CMUSIC_MATH_DOTTED_TIME;					/**< Dotted time multiplier */
		};
	}
}
#endif	// End of CMusicMath.hpp